'use strict';

/**
 * @author Vadym Gidulian, GVIA Group
 */

const canvas = document.getElementById('game');
var ctx;
const SIZE = canvas.width;
const THICK_BORDER = 10;
const WHITE_BORDER = 3*THICK_BORDER;
const BORDER = THICK_BORDER + WHITE_BORDER;
const SQUARE = (SIZE - 2 * BORDER) / 8;
const PIECE = 0.7 * SQUARE;

const FONT = WHITE_BORDER;
const FONT_WIDTH = 0.6 * FONT;

const COLOR_BORDER_THICK = '#000';
const COLOR_BORDER_THIN = '#000';
const COLOR_BORDER_LETTERS = '#fff';
const COLOR_LETTERS = '#000';
const COLOR_SQUARE_BLACK = '#000';
const COLOR_SQUARE_WHITE = '#fff';
const COLOR_PIECE_BLACK = '#000';
const COLOR_PIECE_WHITE = '#fff';

const game = {
    state: '',
    stateData: {},
    turn: 'black',
    pieces: [
        {position: 'A1', color: 'black', isKing: false},
        {position: 'A3', color: 'black', isKing: false},
        {position: 'B2', color: 'black', isKing: false},
        {position: 'C1', color: 'black', isKing: false},
        {position: 'C3', color: 'black', isKing: false},
        {position: 'D2', color: 'black', isKing: false},
        {position: 'E1', color: 'black', isKing: false},
        {position: 'E3', color: 'black', isKing: false},
        {position: 'F2', color: 'black', isKing: false},
        {position: 'G1', color: 'black', isKing: false},
        {position: 'G3', color: 'black', isKing: false},
        {position: 'H2', color: 'black', isKing: false},
        {position: 'A7', color: 'white', isKing: false},
        {position: 'B6', color: 'white', isKing: false},
        {position: 'B8', color: 'white', isKing: false},
        {position: 'C7', color: 'white', isKing: false},
        {position: 'D6', color: 'white', isKing: false},
        {position: 'D8', color: 'white', isKing: false},
        {position: 'E7', color: 'white', isKing: false},
        {position: 'F6', color: 'white', isKing: false},
        {position: 'F8', color: 'white', isKing: false},
        {position: 'G7', color: 'white', isKing: false},
        {position: 'H6', color: 'white', isKing: false},
        {position: 'H8', color: 'white', isKing: false}
    ]
};

gameStart();

// Game

function gameStart() {
    init();
    game.state = 'waiting';
    gameUpdate();

    canvas.addEventListener('mousedown', function (e) {
        var position = getPositionByCoordinates(e.clientX - canvas.offsetLeft, e.clientY - canvas.offsetTop);
        if (position) {
            moveStart(position);
        }
    });
    canvas.addEventListener('mousemove', function (e) {
        moveGoing(e.clientX - canvas.offsetLeft, e.clientY - canvas.offsetTop);
    });
    canvas.addEventListener('mouseup', function (e) {
        var position = getPositionByCoordinates(e.clientX - canvas.offsetLeft, e.clientY - canvas.offsetTop);
        if (position) {
            moveEnd(position);
        }
    });
}

function gameUpdate() {
    drawGameBoard();
    drawPieces();
}

function moveStart(position) {
    if (!getPieceByPosition(position)) return;
    if (game.state == 'moving') return;

    game.state = 'moving';
    game.stateData.piece = getPieceByPosition(position);
    game.stateData.originalPosition = game.stateData.piece.position;
}

function moveGoing(x, y) {
    if (game.state != 'moving') return;

    game.stateData.piece.position = x + ',' + y;
    gameUpdate();
}

function moveEnd(position) {
    if (game.state != 'moving') return;

    if (move(game.stateData.piece, game.stateData.originalPosition, position)) {
        game.stateData.piece.position = position;
        becomeKing(game.stateData.piece);
        nextTurn();
    } else {
        game.stateData.piece.position = game.stateData.originalPosition;
    }
    game.stateData = {};
    game.state = 'waiting';
    gameUpdate();
}

function getPieceByPosition(position) {
    return game.pieces[getPieceIndexByPosition(position)];
}

function getPieceIndexByPosition(position) {
    return game.pieces.map(function (piece) {
        return piece.position;
    }).indexOf(position);
}

// Rules

function move(piece, oldPosition, newPosition) {
    if (piece.color != game.turn) return false;
    if (getPieceByPosition(newPosition)) return false;

    var ox = oldPosition.charCodeAt(0);
    var oy = oldPosition.charCodeAt(1);
    var nx = newPosition.charCodeAt(0);
    var ny = newPosition.charCodeAt(1);

    return piece.color == 'black' && (ny - oy) == 1 && Math.abs(nx - ox) == 1
        || piece.color == 'black' && (ny - oy) == 2 && Math.abs(nx - ox) == 2 && beat(piece, fromCharCodes((ox+nx)/2, (oy+ny)/2))
        || piece.color == 'white' && (oy - ny) == 1 && Math.abs(nx - ox) == 1
        || piece.color == 'white' && (oy - ny) == 2 && Math.abs(nx - ox) == 2 && beat(piece, fromCharCodes((ox+nx)/2, (oy+ny)/2))
        || piece.isKing && piece.color == 'black' && Math.abs(ny - oy) == 1 && Math.abs(nx - ox) == 1
        || piece.isKing && piece.color == 'black' && Math.abs(ny - oy) == 2 && Math.abs(nx - ox) == 2 && beat(piece, fromCharCodes((ox+nx)/2, (oy+ny)/2))
        || piece.isKing && piece.color == 'white' && Math.abs(oy - ny) == 1 && Math.abs(nx - ox) == 1
        || piece.isKing && piece.color == 'white' && Math.abs(oy - ny) == 2 && Math.abs(nx - ox) == 2 && beat(piece, fromCharCodes((ox+nx)/2, (oy+ny)/2));
}

function beat(piece, position) {
    var index = getPieceIndexByPosition(position);
    if (!~index) return false;
    if (piece.color == game.pieces[index].color) return false;
    game.pieces.splice(index, 1);
    return true;
}

function becomeKing(piece) {
    piece.isKing = piece.isKing
        || piece.color == 'black' && piece.position[1] == '8'
        || piece.color == 'white' && piece.position[1] == '1';
}

function nextTurn() {
    game.turn = game.turn == 'black' ? 'white' : 'black';
    console.log(game.turn);
}

// Graphics

function init() {
    ctx = canvas.getContext('2d');
}

function drawGameBoard() {
    // Borders
    ctx.fillStyle = COLOR_BORDER_THICK;
    ctx.fillRect(0, 0, SIZE, SIZE);
    ctx.fillStyle = COLOR_BORDER_LETTERS;
    ctx.fillRect(THICK_BORDER, THICK_BORDER, SIZE - 2*THICK_BORDER, SIZE - 2*THICK_BORDER);
    ctx.fillStyle = COLOR_BORDER_THIN;
    ctx.strokeRect(BORDER, BORDER, SIZE - 2*BORDER, SIZE - 2*BORDER);

    // Squares
    for (var i = 0; i < 8; i++) {
        for (var j = 0; j < 8; j++) {
            if ((i + j) % 2) {
                ctx.fillStyle = COLOR_SQUARE_BLACK;
                ctx.fillRect(BORDER + j*SQUARE, BORDER + i*SQUARE, SQUARE, SQUARE);
            } else {
                ctx.fillStyle = COLOR_SQUARE_WHITE;
                ctx.fillRect(BORDER + j*SQUARE, BORDER + i*SQUARE, SQUARE, SQUARE);
            }
        }
    }

    ctx.font = FONT + 'px serif';
    var letters = ['A', 'B', 'C', 'D', 'E', 'F', 'G', 'H'];
    var numbers = ['1', '2', '3', '4', '5', '6', '7', '8'];
    ctx.fillStyle = COLOR_LETTERS;
    for (i = 0; i < 8; i++) {
        ctx.fillText(letters[i], BORDER+SQUARE/2-FONT_WIDTH/2 + i*SQUARE, BORDER-FONT/6, FONT_WIDTH);
        ctx.fillText(letters[i], BORDER+SQUARE/2-FONT_WIDTH/2 + i*SQUARE, SIZE-THICK_BORDER-FONT/6, FONT_WIDTH);

        ctx.fillText(numbers[7-i], THICK_BORDER+(WHITE_BORDER-FONT_WIDTH)/2, BORDER+SQUARE/2+FONT/2 + i*SQUARE, FONT_WIDTH);
        ctx.fillText(numbers[7-i], SIZE-BORDER+(WHITE_BORDER-FONT_WIDTH)/2, BORDER+SQUARE/2+FONT/2 + i*SQUARE, FONT_WIDTH);
    }
}

function drawPieces() {
    game.pieces.forEach(function (piece) {
        drawPiece(piece.position, piece.color, piece.isKing);
    });
    if (game.stateData.piece) {
        drawPiece(game.stateData.piece.position, game.stateData.piece.color, game.stateData.piece.isKing);
    }
}

function drawPiece(position, color, isKing) {
    var x, y;

    if (position.length == 2) {
        var col = position.toUpperCase().charCodeAt(0) - 65;
        var row = position.toUpperCase().charCodeAt(1) - 49;

        x = BORDER+SQUARE/2 + col*SQUARE;
        y = BORDER+SQUARE/2 + (7-row)*SQUARE;
    } else {
        [x, y] = position.split(',');
        x = +x;
        y = +y;
    }

    switch (color) {
        case 'black':
            ctx.fillStyle = COLOR_PIECE_BLACK;
            ctx.strokeStyle = COLOR_PIECE_WHITE;
            break;
        case 'white':
            ctx.fillStyle = COLOR_PIECE_WHITE;
            ctx.strokeStyle = COLOR_PIECE_BLACK;
    }

    ctx.beginPath();
    ctx.arc(x, y, PIECE/2, 0, 2*Math.PI);
    ctx.fill();

    ctx.beginPath();
    ctx.arc(x, y, PIECE/2, 0, 2*Math.PI);
    ctx.stroke();

    ctx.beginPath();
    ctx.arc(x, y, PIECE/3, 0, 2*Math.PI);
    ctx.stroke();

    if (isKing) {
        switch (color) {
            case 'black':
                ctx.fillStyle = COLOR_PIECE_WHITE;
                break;
            case 'white':
                ctx.fillStyle = COLOR_PIECE_BLACK;
        }

        ctx.beginPath();
        ctx.arc(x, y, PIECE/10, 0, 2*Math.PI);
        ctx.fill();
    }
}

function getPositionByCoordinates(x, y) {
    if (x < BORDER || y < BORDER || x > (SIZE-BORDER) || y > (SIZE-BORDER)) return null;
    var col = Math.floor((x - BORDER) / SQUARE);
    var row = Math.floor((y - BORDER) / SQUARE);

    return String.fromCharCode(col + 65) + String.fromCharCode((7-row) + 49);
}

// Utils

function fromCharCodes() {
    var s = '';
    for (var i = 0; i < arguments.length; i++) {
        s += String.fromCharCode(arguments[i]);
    }
    return s;
}
