'use strict';

module.exports = {
	
	directive: function (options) {
		return {
			bind:             fd('bind',             options),
			inserted:         fd('inserted',         options),
			update:           fd('update',           options),
			componentUpdated: fd('componentUpdated', options),
			unbind:           fd('unbind',           options)
		};
	},
	
	wrap: function (directive, options) {
		if (!directive) return;
		options = options || {};
		
		var originalDef;
		
		if (typeof directive === 'function') {
			originalDef = {
				bind:   directive,
				update: directive
			};
			options.hooks = options.hooks || {bind: true, update: true};
		} else if (typeof directive === 'object') {
			originalDef = directive;
			options.hooks = options.hooks || Object.keys(directive).reduce(function (hooks, hookName) {
				hooks[hookName] = true;
				return hooks;
			}, {});
		}
		
		return {
			bind:             _wrap('bind'),
			inserted:         _wrap('inserted'),
			update:           _wrap('update'),
			componentUpdated: _wrap('componentUpdated'),
			unbind:           _wrap('unbind')
		};
		
		function _wrap(hook) {
			var d = (typeof directive === 'object') ? directive[hook] : directive;
			
			return function (el, binding, vnode, oldVnode) {
				var filteredBinding = getFilteredBinding(binding, originalDef);
				
				fw(hook, options)(el, filteredBinding, vnode, oldVnode, binding);
				if (originalDef[hook]) d(el, filteredBinding, vnode, oldVnode);
			}
		}
	}
	
};

function fd(hook, options) {
	options = options || {};
	
	return function (el, binding, vnode, oldVnode) {
		if ((binding.value && binding.value.disabled) || options.disabled) return;
		
		var hasModifiers = Object.keys(binding.modifiers).length;
		
		var hooks = (binding.value && binding.value.hooks) || (hasModifiers && binding.modifiers) || options.hooks || {};
		var label = (binding.value && binding.value.label) || binding.arg || options.label || '';
		var name  =  binding.name;
		var style = (binding.value && binding.value.style) || options.style || '';
		
		if (Object.keys(hooks).length && !hooks[hook]) return;
		
		log(el, binding, vnode, oldVnode, {
			hook:  hook,
			label: label,
			name:  name,
			style: style
		});
	}
}

function fw(hook, options) {
	options = options || {};
	
	return function (el, binding, vnode, oldVnode, rawBinding) {
		if ((rawBinding.value && rawBinding.value.test && rawBinding.value.test.disabled) || options.disabled) return;
		
		var testModifiers = Object.keys(rawBinding.modifiers).filter(function (modifier) {
			return /^test:/.test(modifier);
		});
		var hasModifiers = testModifiers.length;
		var modifiers = testModifiers.reduce(function (modifiers, modifier) {
			modifiers[modifier.slice(5)] = rawBinding.modifiers[modifier];
			return modifiers;
		}, {});
		
		var argLabel = rawBinding.arg && rawBinding.arg.split(/:?test:/)[1] || '';
		
		var hooks = (rawBinding.value && rawBinding.value.test && rawBinding.value.test.hooks) || (hasModifiers && modifiers) || options.hooks || {};
		var label = (rawBinding.value && rawBinding.value.test && rawBinding.value.test.label) || argLabel || options.label || '';
		var name  =  rawBinding.name;
		var style = (rawBinding.value && rawBinding.value.test && rawBinding.value.test.style) || options.style || '';
		
		if (Object.keys(hooks).length && !hooks[hook]) return;
		
		log(el, binding, vnode, oldVnode, {
			hook:  hook,
			label: label,
			name:  name,
			style: style
		});
	}
}

function getFilteredBinding(rawBinding, def) {
	var binding = {
		arg:        !rawBinding.arg ? rawBinding.arg : rawBinding.arg.split(/:?test:/)[0],
		def:        def,
		modifiers:  getFilteredModifiers(rawBinding.modifiers),
		name:       rawBinding.name,
		rawBinding: rawBinding,
		rawName:    getFilteredRawName(rawBinding.rawName),
		value:      rawBinding.value && rawBinding.value.test && rawBinding.value.test.value || getFilteredValue(rawBinding.value)
	};
	if (!binding.arg) delete binding.arg;
	
	return binding;
	
	function getFilteredModifiers(modifiers) {
		return Object.keys(modifiers).filter(function (mod) {
			return !/^test:/.test(mod);
		}).reduce(function (mods, mod) {
			mods[mod] = rawBinding[mod];
			return mods;
		}, {});
	}
	
	function getFilteredRawName(rawName) {
		var modifiers = rawName.split('.');
		var nameAndArg = modifiers.shift();
		modifiers = modifiers.filter(function (modifier) {
			return !/^test:/.test(modifier);
		});
		
		var groups = /^([A-Za-z0-9-]+)(:([A-Za-z0-9:-]*?)(:test:([A-Za-z0-9]+))?)?$/.exec(nameAndArg);
		
		var name = groups[1];
		var arg  = groups[3];
		
		return name + (arg ? (':' + arg) : '') + (modifiers.length ? ('.' + modifiers.join('.')) : '');
	}
	
	function getFilteredValue(value) {
		if (!value || (typeof value !== 'object')) return value;
		
		var copy = Object.assign({}, value);
		delete copy.test;
		
		return copy;
	}
}

function log(el, binding, vnode, oldVnode, params) {
	console.groupCollapsed('%c' + (params.label ? '['+params.label+'] ' : '') + 'v-' + params.name + ': ' + params.hook, params.style);
	
	console.log(el);
	console.log(binding);
	console.log(vnode);
	if (~['update', 'componentUpdated'].indexOf(params.hook)) console.log(oldVnode);
	
	console.groupEnd();
}
