'use strict';

var observers = new WeakMap();

module.exports = {
	
	bind: function (el, binding) {
		observers.set(el, new MutationObserver(function () {
			stopObserver(el);
			updateDOM(el, binding.value);
			startObserver(el);
		}));
		
		updateDOM(el, binding.value);
		
		startObserver(el);
	},
	
	unbind: function (el) {
		stopObserver(el);
		observers.delete(el);
	}
	
};



function startObserver(el) {
	observers.get(el).observe(el, {
		attributeFilter: ['class', 'style'],
		childList:       true,
		subtree:         true
	});
}

function stopObserver(el) {
	observers.get(el).disconnect();
}



function updateDOM(root, config) {
	Object.keys(config).forEach(function (selector) {
		const qEls = Array.from(root.querySelectorAll(selector));
		if (root.matches(selector)) qEls.unshift(root);
		
		qEls.forEach(function (qEl) {
			applyConfig(qEl, config[selector]);
		});
	});
}

function applyConfig(el, config) {
	config = valueOrFunctionValue(config, el);
	if (!config) return;
	
	if (typeof config === 'object' && !Array.isArray(config)) {
		if (config.class) {
			processAsClassNames(el, config.class);
		}
		if (config.style) {
			applyStyleProperties(el, config.style);
		}
	} else {
		processAsClassNames(el, config);
	}
}

function processAsClassNames(el, config) {
	config = valueOrFunctionValue(config, el);
	if (!config) return;
	
	if (Array.isArray(config)) {
		applyClassNames(el, config);
	} else if (typeof config === 'string') {
		applyClassNames(el, config.split(/\s+/));
	}
}

function applyClassNames(el, classNames) {
	classNames.forEach(function (className) {
		applyClassName(el, className);
	});
}

function applyClassName(el, className) {
	className = valueOrFunctionValue(className, el);
	if (!className) return;
	
	if (/^~.+~$/.test(className)) {
		el.classList.remove(className.substring(1, className.length-1));
	} else {
		el.classList.add(className);
	}
}

function applyStyleProperties(el, properties) {
	properties = valueOrFunctionValue(properties, el);
	if (!properties) return;
	
	Object.keys(properties).forEach(function (property) {
		el.style[property] = valueOrFunctionValue(properties[property], el);
	});
}



function valueOrFunctionValue(value, el) {
	return (typeof value === 'function') ? value(el) : value;
}
