'use strict';

const fs   = require('fs');
const jsvv = require('@gviagroup/jsvv');

const rulesSchema = require('./schemas/rules');

module.exports = function ({headerName = 'X-Token', pathToRules}) {
	const RULES = jsvv(JSON.parse(fs.readFileSync(pathToRules, {encoding: 'utf8'})), rulesSchema, {root: 'rules'});
	
	const TOKEN_HEADER_LC = headerName.toLowerCase();
	
	return async (req, res, next) => {
		const token = req.headers[TOKEN_HEADER_LC] || '';
		
		if (!hasAccess(RULES, token, req.method, req.url)) return res.status(403).send('Token is not valid');
		
		next();
	};
};



function hasAccess(rules, token, method, path) {
	if (!rules.hasOwnProperty(token)) return false;
	const tokenRules = rules[token];
	
	if (typeof tokenRules === 'boolean') {
		return tokenRules;
	} else if (Array.isArray(tokenRules)) {
		for (const tokenRule of tokenRules) {
			if (isMatchesRule(tokenRule, method, path)) return true;
		}
		
		return false;
	} else if (tokenRules && (typeof tokenRules === 'object')) {
		return isMatchesRule(tokenRules, method, path);
	}
	
	return false;
}

function isMatchesRule(rule, method, path) {
	if (rule.methods) {
		if (!rule.methods.map(s => s.toLowerCase()).includes(method.toLowerCase())) return false;
	}
	
	if (rule.paths) {
		for (const p of rule.paths) {
			if (new RegExp(p).test(path)) return true;
		}
		return false;
	}
	
	return true;
}
