'use strict';

const test  = require('ava');
const axios = require('axios');

const server = require('./_server');

test.before(async () => {
	await server.start();
});

test('"false"', async t => {
	const token = 'false';
	
	await testAccess(t, token, 'GET',  '/',       false);
	await testAccess(t, token, 'POST', '/',       false);
	await testAccess(t, token, 'GET',  '/test',   false);
	await testAccess(t, token, 'POST', '/test',   false);
	await testAccess(t, token, 'GET',  '/test/1', false);
	await testAccess(t, token, 'POST', '/test/1', false);
	await testAccess(t, token, 'GET',  '/test/2', false);
	await testAccess(t, token, 'POST', '/test/2', false);
	
	t.pass();
});

test('"true"', async t => {
	const token = 'true';
	
	await testAccess(t, token, 'GET',  '/',       true);
	await testAccess(t, token, 'POST', '/',       true);
	await testAccess(t, token, 'GET',  '/test',   true);
	await testAccess(t, token, 'POST', '/test',   true);
	await testAccess(t, token, 'GET',  '/test/1', true);
	await testAccess(t, token, 'POST', '/test/1', true);
	await testAccess(t, token, 'GET',  '/test/2', true);
	await testAccess(t, token, 'POST', '/test/2', true);
	
	t.pass();
});

test('"42"', async t => {
	const token = 42;
	
	await testAccess(t, token, 'GET',  '/',       true);
	await testAccess(t, token, 'POST', '/',       true);
	await testAccess(t, token, 'GET',  '/test',   true);
	await testAccess(t, token, 'POST', '/test',   true);
	await testAccess(t, token, 'GET',  '/test/1', true);
	await testAccess(t, token, 'POST', '/test/1', true);
	await testAccess(t, token, 'GET',  '/test/2', true);
	await testAccess(t, token, 'POST', '/test/2', true);
	
	t.pass();
});

test('"foo"', async t => {
	const token = 'foo';
	
	await testAccess(t, token, 'GET',  '/',       true);
	await testAccess(t, token, 'POST', '/',       true);
	await testAccess(t, token, 'GET',  '/test',   true);
	await testAccess(t, token, 'POST', '/test',   true);
	await testAccess(t, token, 'GET',  '/test/1', true);
	await testAccess(t, token, 'POST', '/test/1', true);
	await testAccess(t, token, 'GET',  '/test/2', true);
	await testAccess(t, token, 'POST', '/test/2', true);
	
	t.pass();
});

test('" a b c "', async t => {
	const token = ' a b c ';
	
	await testAccess(t, token, 'GET',  '/',       false);
	await testAccess(t, token, 'POST', '/',       false);
	await testAccess(t, token, 'GET',  '/test',   false);
	await testAccess(t, token, 'POST', '/test',   false);
	await testAccess(t, token, 'GET',  '/test/1', false);
	await testAccess(t, token, 'POST', '/test/1', false);
	await testAccess(t, token, 'GET',  '/test/2', false);
	await testAccess(t, token, 'POST', '/test/2', false);
	
	t.pass();
});

test('"a b c"', async t => {
	const token = 'a b c';
	
	await testAccess(t, token, 'GET',  '/',       true);
	await testAccess(t, token, 'POST', '/',       true);
	await testAccess(t, token, 'GET',  '/test',   true);
	await testAccess(t, token, 'POST', '/test',   true);
	await testAccess(t, token, 'GET',  '/test/1', true);
	await testAccess(t, token, 'POST', '/test/1', true);
	await testAccess(t, token, 'GET',  '/test/2', true);
	await testAccess(t, token, 'POST', '/test/2', true);
	
	t.pass();
});

test('"GET@*"', async t => {
	const token = 'GET@*';
	
	await testAccess(t, token, 'GET',  '/',       true);
	await testAccess(t, token, 'POST', '/',       false);
	await testAccess(t, token, 'GET',  '/test',   true);
	await testAccess(t, token, 'POST', '/test',   false);
	await testAccess(t, token, 'GET',  '/test/1', true);
	await testAccess(t, token, 'POST', '/test/1', false);
	await testAccess(t, token, 'GET',  '/test/2', true);
	await testAccess(t, token, 'POST', '/test/2', false);
	
	t.pass();
});

test('"GET@/test"', async t => {
	const token = 'GET@/test';
	
	await testAccess(t, token, 'GET',  '/',       false);
	await testAccess(t, token, 'POST', '/',       false);
	await testAccess(t, token, 'GET',  '/test',   true);
	await testAccess(t, token, 'POST', '/test',   false);
	await testAccess(t, token, 'GET',  '/test/1', true);
	await testAccess(t, token, 'POST', '/test/1', false);
	await testAccess(t, token, 'GET',  '/test/2', true);
	await testAccess(t, token, 'POST', '/test/2', false);
	
	t.pass();
});

test('"*@/test"', async t => {
	const token = '*@/test';
	
	await testAccess(t, token, 'GET',  '/',       false);
	await testAccess(t, token, 'POST', '/',       false);
	await testAccess(t, token, 'GET',  '/test',   true);
	await testAccess(t, token, 'POST', '/test',   true);
	await testAccess(t, token, 'GET',  '/test/1', true);
	await testAccess(t, token, 'POST', '/test/1', true);
	await testAccess(t, token, 'GET',  '/test/2', true);
	await testAccess(t, token, 'POST', '/test/2', true);
	
	t.pass();
});

test('"GET@/test/1:POST@test/2"', async t => {
	const token = 'GET@/test/1:POST@test/2';
	
	await testAccess(t, token, 'GET',  '/',       false);
	await testAccess(t, token, 'POST', '/',       false);
	await testAccess(t, token, 'GET',  '/test',   false);
	await testAccess(t, token, 'POST', '/test',   false);
	await testAccess(t, token, 'GET',  '/test/1', true);
	await testAccess(t, token, 'POST', '/test/1', false);
	await testAccess(t, token, 'GET',  '/test/2', false);
	await testAccess(t, token, 'POST', '/test/2', true);
	
	t.pass();
});

test('"toString"', async t => {
	const token = 'toString';
	
	await testAccess(t, token, 'GET',  '/',      false);
	await testAccess(t, token, 'POST', '/',      false);
	await testAccess(t, token, 'GET',  '/test',  false);
	await testAccess(t, token, 'POST', '/test',  false);
	await testAccess(t, token, 'GET',  '/test1', false);
	await testAccess(t, token, 'POST', '/test1', false);
	await testAccess(t, token, 'GET',  '/test2', false);
	await testAccess(t, token, 'POST', '/test2', false);
	
	t.pass();
});

test('<none>', async t => {
	const token = undefined;
	
	await testAccess(t, token, 'GET',  '/',      false);
	await testAccess(t, token, 'POST', '/',      true);
	await testAccess(t, token, 'GET',  '/test',  false);
	await testAccess(t, token, 'POST', '/test',  true);
	await testAccess(t, token, 'GET',  '/test1', false);
	await testAccess(t, token, 'POST', '/test1', true);
	await testAccess(t, token, 'GET',  '/test2', false);
	await testAccess(t, token, 'POST', '/test2', true);
	
	t.pass();
});

test('""', async t => {
	const token = '';
	
	await testAccess(t, token, 'GET',  '/',      false);
	await testAccess(t, token, 'POST', '/',      true);
	await testAccess(t, token, 'GET',  '/test',  false);
	await testAccess(t, token, 'POST', '/test',  true);
	await testAccess(t, token, 'GET',  '/test1', false);
	await testAccess(t, token, 'POST', '/test1', true);
	await testAccess(t, token, 'GET',  '/test2', false);
	await testAccess(t, token, 'POST', '/test2', true);
	
	t.pass();
});

test('" "', async t => {
	const token = ' ';
	
	await testAccess(t, token, 'GET',  '/',      false);
	await testAccess(t, token, 'POST', '/',      true);
	await testAccess(t, token, 'GET',  '/test',  false);
	await testAccess(t, token, 'POST', '/test',  true);
	await testAccess(t, token, 'GET',  '/test1', false);
	await testAccess(t, token, 'POST', '/test1', true);
	await testAccess(t, token, 'GET',  '/test2', false);
	await testAccess(t, token, 'POST', '/test2', true);
	
	t.pass();
});

test.after.always(async () => {
	await server.stop();
});



async function testAccess(t, token, method, path, shouldHasAccess) {
	try {
		const {data} = await axios({
			method,
			url: `http://api${path}`,
			headers: {
				...((token !== undefined) ? {'X-Token': token} : {})
			}
		});
		
		if (!shouldHasAccess) t.fail(`Should NOT has access to ${path} with ${method}`);
		
		t.is(data, 'Hello');
	} catch (e) {
		if (e.response.status === 403) {
			if (shouldHasAccess) t.fail(`Should has access to ${path} with ${method}`);
		} else throw e;
	}
}
