'use strict';

const fs = require('fs');

const test     = require('ava');
const axios    = require('axios');
const del      = require('del');
const FormData = require('form-data');
const mkdirp   = require('mkdirp');

const releasesModel = require('../../src/models/releases')();
const server        = require('./_server');

test.serial.before(async t => {
	del.sync('/files/**', {dot: true, force: true});
	mkdirp.sync(releasesModel.TMP_FILES_DIR);
	
	t.deepEqual(fs.readdirSync(releasesModel.FILES_DIR), ['.tmp']);
	
	await server.start();
});

test.serial('There should be no releases', async t => {
	const {data: releases} = await axios.get('http://api/releases');
	
	t.deepEqual(releases, []);
});

test.serial('There should be no latest release', async t => {
	try {
		await axios.get('http://api/latest');
	} catch (e) {
		t.is(e.response.status, 404);
	}
});

test.serial('There should be no latest release info', async t => {
	try {
		await axios.get('http://api/latest/info');
	} catch (e) {
		t.is(e.response.status, 404);
	}
});

test.serial('Create release', async t => {
	const formData = getFormData({fileName: 'file.txt'});
	
	const {data: info} = await axios.put(`http://api/1.0.0`, formData, {
		headers: formData.getHeaders()
	});
	testReleaseInfo(t, info, {
		version:  '1.0.0',
		fileName: 'file.txt'
	});
});

test.serial('Try to create release w/ invalid SemVer version', async t => {
	const formData = getFormData({fileName: 'file.txt'});
	
	try {
		await axios.put(`http://api/1`, formData, {
			headers: formData.getHeaders()
		});
	} catch (e) {
		t.is(e.response.status, 400);
	}
});

test.serial('Try to create release w/ empty body', async t => {
	try {
		await axios.put(`http://api/1`);
	} catch (e) {
		t.is(e.response.status, 400);
	}
});

test.serial('Create release specifying different file name', async t => {
	const formData = getFormData({fileName: 'file.txt'});
	
	const {data: info} = await axios.put(`http://api/1.1.0`, formData, {
		headers: {
			...formData.getHeaders(),
			'X-File-Name': 'release.txt'
		}
	});
	testReleaseInfo(t, info, {
		version:  '1.1.0',
		fileName: 'release.txt'
	});
});

test.serial('Create private release', async t => {
	const formData = getFormData({fileName: 'file.txt'});
	
	const {data: info} = await axios.put(`http://api/2.0.0-beta`, formData, {
		headers: {
			...formData.getHeaders(),
			'X-Private': true
		}
	});
	testReleaseInfo(t, info, {
		version:  '2.0.0-beta',
		fileName: 'file.txt'
	});
});

test.serial('Update release', async t => {
	const formData = getFormData({fileName: 'release.txt'});
	
	const {data: info} = await axios.put(`http://api/1.0.0`, formData, {
		headers: formData.getHeaders()
	});
	testReleaseInfo(t, info, {
		version:  '1.0.0',
		fileName: 'release.txt'
	});
});

test.serial('Get release info', async t => {
	const {data: info} = await axios.get('http://api/1.0.0/info');
	testReleaseInfo(t, info, {
		version:  '1.0.0',
		fileName: 'release.txt'
	});
});

test.serial('Download release', async t => {
	const response = await axios.get('http://api/1.0.0');
	
	t.is(response.headers['content-disposition'], 'attachment; filename="release.txt"');
	t.is(response.headers['content-length'],      '13');
	
	t.is(response.data, 'Hello, World!');
});

test.serial('Get latest release info', async t => {
	const {data: info} = await axios.get('http://api/latest/info');
	testReleaseInfo(t, info, {
		version:  '1.1.0',
		fileName: 'release.txt'
	});
});

test.serial('Download latest release', async t => {
	const response = await axios.get('http://api/latest');
	
	t.is(response.headers['content-disposition'], 'attachment; filename="release.txt"');
	t.is(response.headers['content-length'],      '13');
	
	t.is(response.data, 'Hello, World!');
});

test.serial('Get release info of invalid SemVer version', async t => {
	try {
		await axios.get('http://api/1/info');
	} catch (e) {
		t.is(e.response.status, 400);
	}
});

test.serial('Download release of invalid SemVer version', async t => {
	try {
		await axios.get('http://api/1');
	} catch (e) {
		t.is(e.response.status, 400);
	}
});

test.serial('Get all releases', async t => {
	const {data: releases} = await axios.get('http://api/releases');
	const expectedReleases = [
		{
			version:  '2.0.0-beta',
			fileName: 'file.txt',
			private:  true
		}, {
			version:  '1.1.0',
			fileName: 'release.txt'
		}, {
			version:  '1.0.0',
			fileName: 'release.txt'
		}
	];
	
	t.is(releases.length, expectedReleases.length);
	for (let i = 0; i < releases.length; i++) {
		testReleaseInfo(t, releases[i], expectedReleases[i]);
	}
});

test.serial('Delete release', async t => {
	const {data: info} = await axios.delete('http://api/1.0.0');
	testReleaseInfo(t, info, {
		version:  '1.0.0',
		fileName: 'release.txt'
	});
});

test.serial('Delete private release', async t => {
	const {data: info} = await axios.delete('http://api/2.0.0-beta');
	testReleaseInfo(t, info, {
		version:  '2.0.0-beta',
		fileName: 'file.txt'
	});
});

test.serial('Delete last release', async t => {
	const {data: info} = await axios.delete('http://api/1.1.0');
	testReleaseInfo(t, info, {
		version:  '1.1.0',
		fileName: 'release.txt'
	});
});

test.serial('There should be no releases after deletion of all releases', async t => {
	const {data: releases} = await axios.get('http://api/releases');
	
	t.deepEqual(releases, []);
});

test.serial('There should be no latest release after deletion of all releases', async t => {
	try {
		await axios.get('http://api/latest');
	} catch (e) {
		t.is(e.response.status, 404);
	}
});

test.serial('There should be no latest release info after deletion of all releases', async t => {
	try {
		await axios.get('http://api/latest/info');
	} catch (e) {
		t.is(e.response.status, 404);
	}
});

test.serial('Get release info of non-existing version', async t => {
	try {
		await axios.get('http://api/1.0.0/info');
	} catch (e) {
		t.is(e.response.status, 404);
	}
});

test.serial('Download release of non-existing version', async t => {
	try {
		await axios.get('http://api/1.0.0');
	} catch (e) {
		t.is(e.response.status, 404);
	}
});

test.serial('Delete non-existing release', async t => {
	const {data: info} = await axios.delete('http://api/1.0.0');
	
	t.falsy(info);
});

test.serial.after.always(async () => {
	await server.stop();
});

test.serial.after(async t => {
	del.sync('/files/**', {dot: true, force: true});
	
	t.deepEqual(fs.readdirSync(releasesModel.FILES_DIR), []);
});



function getFormData({fileName}) {
	const formData = new FormData();
	formData.append('file', 'Hello, World!', fileName);
	
	return formData;
}

function testReleaseInfo(t, actual, expected) {
	t.truthy(actual);
	t.is(actual.version,  expected.version);
	t.is(actual.fileName, expected.fileName);
	t.is(actual.fileSize, 13);
	t.true(Date.now() - new Date(actual.publishDate) < 100);
}
