'use strict';

const fs   = require('fs');
const path = require('path');

const test   = require('ava');
const del    = require('del');
const mkdirp = require('mkdirp');

const releasesModel = require('../../src/models/releases')();

test.serial.before(t => {
	del.sync('/files/**', {dot: true, force: true});
	
	t.deepEqual(fs.readdirSync(releasesModel.FILES_DIR), []);
});

test.serial('There should be no releases', async t => {
	const info = await releasesModel.getAllReleases();
	t.deepEqual(info, []);
});

test.serial('There should be no latest release', async t => {
	const info = await releasesModel.getReleaseInfo({version: 'latest'});
	t.is(info, null);
	
	t.is(await releasesModel.getReleaseFilePath({version: 'latest'}), null);
});

test.serial('Create release', async t => {
	const filePath = createTestFile();

	const info = await releasesModel.saveRelease({
		version: '1.0.0',
		filePath,
		fileName: 'file.txt'
	});
	testReleaseInfo(t, info, {
		version:  '1.0.0',
		fileName: 'file.txt'
	});
});

test.serial('Create private release', async t => {
	const filePath = createTestFile();
	
	const info = await releasesModel.saveRelease({
		version: '2.0.0-beta',
		filePath,
		fileName:  'file.txt',
		isPrivate: true
	});
	testReleaseInfo(t, info, {
		version:  '2.0.0-beta',
		fileName: 'file.txt',
		private:  true
	});
});

test.serial('Update release', async t => {
	const filePath = createTestFile();
	
	const info = await releasesModel.saveRelease({
		version: '1.0.0',
		filePath,
		fileName: 'release.txt'
	});
	testReleaseInfo(t, info, {
		version:  '1.0.0',
		fileName: 'release.txt'
	});
});

test.serial('Get release info', async t => {
	const info = await releasesModel.getReleaseInfo({version: '1.0.0'});
	testReleaseInfo(t, info, {
		version:  '1.0.0',
		fileName: 'release.txt'
	});
	
	t.is(await releasesModel.getReleaseFilePath({version: '1.0.0'}),
	     path.join(releasesModel.FILES_DIR, '1.0.0', 'release.txt'));
});

test.serial('Temp folder should be empty', t => {
	t.deepEqual(fs.readdirSync(releasesModel.TMP_FILES_DIR), []);
});

test.serial('Get all releases', async t => {
	const releases = await releasesModel.getAllReleases();
	const expectedReleases = [
		{
			version:  '2.0.0-beta',
			fileName: 'file.txt',
			private:  true
		}, {
			version:  '1.0.0',
			fileName: 'release.txt'
		}
	];
	
	t.is(releases.length, expectedReleases.length);
	for (let i = 0; i < releases.length; i++) {
		testReleaseInfo(t, releases[i], expectedReleases[i]);
	}
});

test.serial('Delete release', async t => {
	const info = await releasesModel.deleteRelease({version: '1.0.0'});
	testReleaseInfo(t, info, {
		version:  '1.0.0',
		fileName: 'release.txt'
	});
});

test.serial('Delete private release', async t => {
	const info = await releasesModel.deleteRelease({version: '2.0.0-beta'});
	testReleaseInfo(t, info, {
		version:  '2.0.0-beta',
		fileName: 'file.txt',
		private:  true
	});
});

test.serial('There should be no releases after deletion of all releases', async t => {
	const info = await releasesModel.getAllReleases();
	t.deepEqual(info, []);
});

test.serial('There should be no latest release after deletion of all releases', async t => {
	const info = await releasesModel.getReleaseInfo({version: 'latest'});
	t.is(info, null);
	
	t.is(await releasesModel.getReleaseFilePath({version: 'latest'}), null);
});

test.serial('Releases folder should be empty', t => {
	t.deepEqual(fs.readdirSync(releasesModel.FILES_DIR), ['.tmp']);
});

test.serial('Get non-existing release info', async t => {
	const info = await releasesModel.getReleaseInfo({version: '1.0.0'});
	t.is(info, null);
	
	t.is(await releasesModel.getReleaseFilePath({version: '1.0.0'}), null);
});

test.serial('Delete non-existing release', async t => {
	const info = await releasesModel.deleteRelease({version: '1.0.0'});
	t.is(info, null);
});

test.serial.after(t => {
	del.sync('/files/**', {dot: true, force: true});
	
	t.deepEqual(fs.readdirSync(releasesModel.FILES_DIR), []);
});



function createTestFile() {
	mkdirp.sync(releasesModel.TMP_FILES_DIR);
	
	const randomFileName = `${Math.ceil(Math.random() * Number.MAX_SAFE_INTEGER).toString(36)}.txt`;
	const filePath       = path.join(releasesModel.TMP_FILES_DIR, randomFileName);
	
	fs.writeFileSync(filePath, 'Hello, World!');
	
	return filePath;
}

function testReleaseInfo(t, actual, expected) {
	t.truthy(actual);
	t.is(actual.version,  expected.version);
	t.is(actual.fileName, expected.fileName);
	t.is(actual.fileSize, 13);
	t.is(actual.private,  expected.private);
	t.true(Date.now() - new Date(actual.publishDate) < 100);
}
