'use strict';

const path = require('path');

const express                                         = require('express');
const multer                                          = require('multer');
const jsvv                                            = require('@gviagroup/jsvv');
const {handle, methodNotAllowed, SYMBOLS: {IS_EMPTY}} = require('../utils/api');
const {AppError}                                      = require('../utils/errors');

const reqVersionSchema     = require('../schemas/request-version');
const resReleaseInfoSchema = require('../schemas/response-release-info');

module.exports = function ({filesDir, tmpFilesDir} = {}) {
	const releasesModel = require('../models/releases')({filesDir, tmpFilesDir});
	
	const router = express.Router();
	const methodNotAllowedHandler = methodNotAllowed(router);
	
	const fileGetHandler = handle(req => {
			req.params.version = jsvv(req.params.version, reqVersionSchema, {root: 'version'});
		},
		async (req, res, next) => {
			const filePath = await releasesModel.getReleaseFilePath({version: req.params.version});
			if (!filePath) return null;
			
			req.url = path.relative(releasesModel.FILES_DIR, filePath);
			next();
		});
	const latestHandler = (req, res, next) => {
		req.params.version = 'latest';
		next();
	};
	const staticHandler = express.static(releasesModel.FILES_DIR);
	
	const upload = multer({
		dest: releasesModel.TMP_FILES_DIR
	});
	
	
	
	router.route('/latest')
		.head(latestHandler, fileGetHandler, staticHandler)
		.get( latestHandler, fileGetHandler, staticHandler)
		.all(methodNotAllowedHandler);
	
	router.route('/latest/info')
		.get(handle(undefined,
			() => releasesModel.getReleaseInfo({version: 'latest'}),
			response => jsvv(response, resReleaseInfoSchema, {root: 'releaseInfo'})))
		.all(methodNotAllowedHandler);
	
	router.route('/releases')
		.get(handle(undefined,
			() => releasesModel.getAllReleases(),
			response => jsvv(response, {
				type:     Array,
				required: true,
				item:     resReleaseInfoSchema
			}, {root: 'releases'})))
		.all(methodNotAllowedHandler);
	
	router.route('/:version')
		.head(fileGetHandler, staticHandler)
		.get( fileGetHandler, staticHandler)
		.put(upload.single('file'), handle(req => {
				req.params.version = jsvv(req.params.version, reqVersionSchema, {root: 'version'});
				jsvv(req.headers['content-type'], {type: String, required: true, pattern: /^multipart\/form-data/}, {root: 'headers.Content-Type'});
				
				if (!req.file || !req.file.size || !req.file.path) throw new AppError('Body is empty', 400);
			},
			req => releasesModel.saveRelease({
				version:   req.params.version,
				filePath:  req.file.path,
				fileName:  req.headers['x-file-name'] || req.file.originalname,
				isPrivate: req.headers.hasOwnProperty('x-private')
			}),
			response => jsvv(response, resReleaseInfoSchema, {root: 'releaseInfo'})))
		.delete(handle(req => {
				req.params.version = jsvv(req.params.version, reqVersionSchema, {root: 'version'});
			},
			async (req, res) => {
				const info = await releasesModel.deleteRelease({version: req.params.version});
				
				if (!info) res[IS_EMPTY] = true;
				return info;
			},
			response => jsvv(response, resReleaseInfoSchema, {root: 'releaseInfo'})))
		.all(methodNotAllowedHandler);
	
	router.route('/:version/info')
		.get(handle(req => {
				req.params.version = jsvv(req.params.version, reqVersionSchema, {root: 'version'});
			},
			req => releasesModel.getReleaseInfo({version: req.params.version}),
			response => jsvv(response, resReleaseInfoSchema, {root: 'releaseInfo'})))
		.all(methodNotAllowedHandler);
	
	return router;
};
