'use strict';

const fs          = require('fs');
const path        = require('path');
const {promisify} = require('util');

const del    = require('del');
const mkdirp = require('mkdirp');
const semver = require('semver');

const fsRename    = promisify(fs.rename);
const fsStat      = promisify(fs.stat);
const fsReadDir   = promisify(fs.readdir);
const fsReadFile  = promisify(fs.readFile);
const fsWriteFile = promisify(fs.writeFile);
const mkdirpAsync = promisify(mkdirp);

const RELEASE_INFO_FILENAME = 'info.json';

module.exports = function ({filesDir = '/files', tmpFilesDir = path.join(filesDir, '.tmp')} = {}) {
	const FILES_DIR     = filesDir;
	const TMP_FILES_DIR = tmpFilesDir;
	
	return {
		FILES_DIR,
		TMP_FILES_DIR,
		
		deleteRelease,
		getAllReleases,
		getReleaseFilePath,
		getReleaseInfo,
		saveRelease
	};
	
	async function deleteRelease({version}) {
		const info = await getReleaseInfo({version});
		
		await del(getReleaseDir({version}), {force: true});
		
		return info;
	}
	
	async function determineLatestVersion() {
		const versions = await getAllVersions();
		
		for (const version of versions) {
			const info = await getReleaseInfo({version});
			if (!info.private) return version;
		}
		return null;
	}
	
	async function isVersionExists({version}) {
		if (!version) return false;
		
		try {
			await fsStat(getReleaseDir({version}));
			return true;
		} catch (e) {
			if (e.code === 'ENOENT') return false;
			else throw e;
		}
	}
	
	async function getAllReleases() {
		return Promise.all((await getAllVersions())
			.map(version => getReleaseInfo({version})));
	}
	
	async function getAllVersions() {
		return (await fsReadDir(FILES_DIR))
			.filter(s => !s.startsWith('.'))
			.sort(semver.rcompare);
	}
	
	function getReleaseDir({version}) {
		return path.join(FILES_DIR, version);
	}
	
	async function getReleaseFilePath({version}) {
		if (version === 'latest') version = await determineLatestVersion();
		
		const info = await getReleaseInfo({version});
		return info
			? path.join(getReleaseDir({version}), info.fileName)
			: null;
	}
	
	async function getReleaseInfo({version}) {
		if (version === 'latest') version = await determineLatestVersion();
		
		return (await isVersionExists({version}))
			? JSON.parse(await fsReadFile(path.join(getReleaseDir({version}), RELEASE_INFO_FILENAME), {encoding: 'utf8'}))
			: null;
	}
	
	async function saveRelease({version, filePath, fileName, isPrivate}) {
		const releaseDir  = getReleaseDir({version});
		const newFilePath = path.join(releaseDir, fileName);
		
		await deleteRelease({version});
		await mkdirpAsync(releaseDir);
		
		await fsRename(filePath, newFilePath);
		await fsWriteFile(path.join(releaseDir, RELEASE_INFO_FILENAME), JSON.stringify({
			version,
			publishDate: new Date(),
			fileName,
			fileSize: (await fsStat(newFilePath)).size,
			private:  !!isPrivate || undefined
		}));
		
		return getReleaseInfo({version});
	}
};
