'use strict';

const fs   = require('fs');
const path = require('path');
const test = require('ava');

const {objToObjb, objbToObj} = require('./_lib');

const HEADER = [0x1, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0];

test('Empty', t => {
	t.deepEqual(objToObjb(''), new Uint8Array(HEADER).buffer);
});

test('Ignore comments', t => {
	const source = `
	# Comment
	`;
	
	t.deepEqual(objToObjb(source), new Uint8Array(HEADER).buffer);
});

test('v[1]', t => {
	t.deepEqual(objbToObj(objToObjb('v 1 2 3')), `v 1 2 3
`);
});

test('v[2]', t => {
	t.deepEqual(objbToObj(objToObjb(`
		v 1 2 3
		v 4 5 6
	`)), `v 1 2 3
v 4 5 6
`);
});

test('v empty', t => {
	t.deepEqual(objbToObj(objToObjb(`v`)), `v 0 0 0\n`);
});

test('v(5)', t => {
	t.deepEqual(objbToObj(objToObjb(`v 1 2 3 4 5`)), `v 1 2 3\n`);
});

test('vn[1]', t => {
	t.deepEqual(objbToObj(objToObjb(`vn 1 2 3`)), `vn 1 2 3
`);
});

test('vn[2]', t => {
	t.deepEqual(objbToObj(objToObjb(`vn 1 2 3
		vn 4 5 6
	`)), `vn 1 2 3
vn 4 5 6
`);
});

test('vn empty', t => {
	t.deepEqual(objbToObj(objToObjb(`vn`)), `vn 0 0 0\n`);
});

test('vn(5)', t => {
	t.deepEqual(objbToObj(objToObjb(`vn 1 2 3 4 5`)), `vn 1 2 3\n`);
});

test('vt(u,v,w)[1]', t => {
	t.deepEqual(objbToObj(objToObjb(`vt 1 2 3`)), `vt 1 2 3
`);
});

test('vt(u,v,w)[2]', t => {
	t.deepEqual(objbToObj(objToObjb(`vt 1 2 3
		vt 4 5 6
	`)), `vt 1 2 3
vt 4 5 6
`);
});

test('vt(u,v)[1]', t => {
	t.deepEqual(objbToObj(objToObjb(`vt 1 2`)), `vt 1 2 0
`);
});

test('vt(u,v)[2]', t => {
	t.deepEqual(objbToObj(objToObjb(`vt 1 2
		vt 3 4
	`)), `vt 1 2 0
vt 3 4 0
`);
});

test('vt(u)[1]', t => {
	t.deepEqual(objbToObj(objToObjb(`vt 1`)), `vt 1 0 0
`);
});

test('vt(u)[2]', t => {
	t.deepEqual(objbToObj(objToObjb(`vt 1
		vt 2
	`)), `vt 1 0 0
vt 2 0 0
`);
});

test('vt empty', t => {
	t.deepEqual(objbToObj(objToObjb(`vt`)), `vt 0 0 0\n`);
});

test('vt(5)', t => {
	t.deepEqual(objbToObj(objToObjb(`vt 1 2 3 4 5`)), `vt 1 2 3\n`);
});

test('g', t => {
	t.deepEqual(objbToObj(objToObjb(`g group`)), `g group
`);
});

test('g[2]', t => {
	t.deepEqual(objbToObj(objToObjb(`g group1 group2`)), `g group1 group2
`);
});

test('g empty', t => {
	t.deepEqual(objbToObj(objToObjb(`g`)), `g \n`);
});

test('o', t => {
	t.deepEqual(objbToObj(objToObjb(`o name`)), `o name
`);
});

test('o[2]', t => {
	t.deepEqual(objbToObj(objToObjb(`o name1 name2`)), `o name1 name2
`);
});

test('o empty', t => {
	t.deepEqual(objbToObj(objToObjb(`o`)), `o \n`);
});

test('s <number>', t => {
	t.deepEqual(objbToObj(objToObjb(`s 1`)), `s 1
`);
});

test('s off', t => {
	t.deepEqual(objbToObj(objToObjb(`s off`)), `s off
`);
});

test('s empty', t => {
	t.deepEqual(objbToObj(objToObjb(`s`)), `s \n`);
});

test('f(3)(v,vt,vn)[1]', t => {
	t.deepEqual(objbToObj(objToObjb(`f 1/2/3 4/5/6 7/8/9`)), `f 1/2/3 4/5/6 7/8/9
`);
});

test('f(3)(v,vt)[1]', t => {
	t.deepEqual(objbToObj(objToObjb(`f 1/2 3/4 5/6`)), `f 1/2 3/4 5/6
`);
});

test('f(3)(v,vn)[1]', t => {
	t.deepEqual(objbToObj(objToObjb(`f 1//2 3//4 5//6`)), `f 1//2 3//4 5//6
`);
});

test('f(3)(v)[1]', t => {
	t.deepEqual(objbToObj(objToObjb(`f 1 2 3`)), `f 1 2 3
`);
});

test('f(4)(v,vt,vn)[1]', t => {
	t.deepEqual(objbToObj(objToObjb(`f 1/2/3 4/5/6 7/8/9 10/11/12`)), `f 1/2/3 4/5/6 7/8/9 10/11/12
`);
});

test('f(4)(v,vt)[1]', t => {
	t.deepEqual(objbToObj(objToObjb(`f 1/2 3/4 5/6 7/8`)), `f 1/2 3/4 5/6 7/8
`);
});

test('f(4)(v,vn)[1]', t => {
	t.deepEqual(objbToObj(objToObjb(`f 1//2 3//4 5//6 7//8`)), `f 1//2 3//4 5//6 7//8
`);
});

test('f(4)(v)[1]', t => {
	t.deepEqual(objbToObj(objToObjb(`f 1 2 3 4`)), `f 1 2 3 4
`);
});

test('f(3)(v,vt,vn)[2]', t => {
	t.deepEqual(objbToObj(objToObjb(`f 1/2/3 4/5/6 7/8/9
		f 10/11/12 13/14/15 16/17/18
	`)), `f 1/2/3 4/5/6 7/8/9
f 10/11/12 13/14/15 16/17/18
`);
});

test('f(3)(v,vt)[2]', t => {
	t.deepEqual(objbToObj(objToObjb(`f 1/2 3/4 5/6
		f 7/8 9/10 11/12
	`)), `f 1/2 3/4 5/6
f 7/8 9/10 11/12
`);
});

test('f(3)(v,vn)[2]', t => {
	t.deepEqual(objbToObj(objToObjb(`f 1//2 3//4 5//6
		f 7//8 9//10 11//12
	`)), `f 1//2 3//4 5//6
f 7//8 9//10 11//12
`);
});

test('f(3)(v)[2]', t => {
	t.deepEqual(objbToObj(objToObjb(`f 1 2 3
		f 4 5 6
	`)), `f 1 2 3
f 4 5 6
`);
});

test('f(4)(v,vt,vn)[2]', t => {
	t.deepEqual(objbToObj(objToObjb(`f 1/2/3 4/5/6 7/8/9 10/11/12
		f 13/14/15 16/17/18 19/20/21 22/23/24
	`)), `f 1/2/3 4/5/6 7/8/9 10/11/12
f 13/14/15 16/17/18 19/20/21 22/23/24
`);
});

test('f(4)(v,vt)[2]', t => {
	t.deepEqual(objbToObj(objToObjb(`f 1/2 3/4 5/6 7/8
		f 9/10 11/12 13/14 15/16
	`)), `f 1/2 3/4 5/6 7/8
f 9/10 11/12 13/14 15/16
`);
});

test('f(4)(v,vn)[2]', t => {
	t.deepEqual(objbToObj(objToObjb(`f 1//2 3//4 5//6 7//8
		f 9//10 11//12 13//14 15//16
	`)), `f 1//2 3//4 5//6 7//8
f 9//10 11//12 13//14 15//16
`);
});

test('f(4)(v)[2]', t => {
	t.deepEqual(objbToObj(objToObjb(`f 1 2 3 4
		f 5 6 7 8
	`)), `f 1 2 3 4
f 5 6 7 8
`);
});

test('Unknown file version', t => {
	t.throws(() => {
		objbToObj(new Uint8Array([2, 0, 0, 0, 0, 0, 0, 0]).buffer);
	}, SyntaxError);
});

test('Invalid data structure', t => {
	t.throws(() => {
		objbToObj(new Uint8Array([1, 0]).buffer);
	}, SyntaxError);
});

test('Should not stuck', t => {
	t.timeout(1000);
	
	t.throws(() => {
		objbToObj(new Uint8Array([...HEADER, 42]).buffer);
	}, SyntaxError);
});

test('Teapot (float32)', t => {
	const source = fs.readFileSync(path.join(__dirname, '_teapot.obj'), {encoding: 'utf8'});
	
	const buffer = objToObjb(source);
	objbToObj(buffer);
	
	t.pass();
});

test('Teapot (float16)', t => {
	const source = fs.readFileSync(path.join(__dirname, '_teapot.obj'), {encoding: 'utf8'});
	
	const buffer = objToObjb(source, {useFloat16: true});
	objbToObj(buffer);
	
	t.pass();
});
