'use strict';

const {getFloat16, setFloat16} = require('@petamoriken/float16');

const KEYWORDS = {
	v:  0x1,
	vn: 0x2,
	vt: 0x3,
	g:  0x4,
	o:  0x5,
	s:  0x6,
	f:  0x7
};

module.exports = {
	
	objToObjb,
	objbToObj
	
};

function objToObjb(source, {useFloat16 = false} = {}) {
	const buffer = new ArrayBuffer(Math.max(1024, source.length));
	const view   = new DataView(buffer);
	
	view.setUint16(0, 0x0100);
	if (useFloat16) view.setUint8(2, 1);
	let offset = 8;
	
	const lines = source
		.split(/\r\n|\r|\n/)
		.map(line => line.trim())
		.filter(Boolean);
	
	let currentSection       = null;
	let currentSectionLength = 0;
	let currentSectionOffset = null;
	
	let vCount  = 0;
	let vtCount = 0;
	let vnCount = 0;
	
	for (const line of lines) {
		const [keyword, ...args] = line.split(/\s+/);
		
		writeSection(keyword);
		
		switch (keyword) {
			case 'f':
				view.setUint8(offset++, args.length);
				
				const bits = Math.max(8, 2**Math.ceil(Math.log2(Math.log2(Math.max(vCount, vtCount, vnCount)+1))));
				view.setUint8(offset++, bits);
				
				for (const arg of args) {
					const nums = arg.split('/');
					for (let i = 0; i < 3; i++) {
						view[`setUint${bits}`](offset, +(nums[i] || 0));
						offset += bits / 8;
					}
				}
				
				currentSectionLength++;

				break;
			case 'g':
			case 'o':
			case 's':
				writeString(args.join(' '));

				break;
			case 'v':
				writeTriplet(args);
				vCount++;
				
				break;
			case 'vn':
				writeTriplet(args);
				vnCount++;
				
				break;
			case 'vt':
				writeTriplet(args);
				vtCount++;
				
				break;
		}
	}
	
	writeSection(null);
	
	return buffer.slice(0, offset);
	
	
	
	function writeSection(keyword) {
		if (currentSection === keyword) return;
		
		if (currentSection) view.setUint32(currentSectionOffset+1, currentSectionLength);
		
		if (!Object.keys(KEYWORDS).includes(keyword)) return;
		currentSection       = keyword;
		currentSectionLength = 0;
		currentSectionOffset = offset;
		view.setUint8(offset, KEYWORDS[keyword]);
		offset += 5;
	}
	
	function writeString(string) {
		for (let i = 0; i < string.length; i++) {
			view.setUint16(offset, string.charCodeAt(i));
			offset += 2;
		}
		
		currentSectionLength += string.length;
	}
	
	function writeTriplet(args) {
		for (let i = 0; i < 3; i++) {
			if (useFloat16) {
				setFloat16(view, offset, +args[i] || 0);
				offset += 2;
			} else {
				view.setFloat32(offset, +args[i] || 0);
				offset += 4;
			}
		}
		currentSectionLength++;
	}
}

function objbToObj(buffer) {
	const view = new DataView(buffer);
	let offset = 0;
	let source = '';
	
	try {
		if (view.getUint16(0) !== 0x0100) throw new SyntaxError('Unknown version');
		const useFloat16 = !!(view.getUint8(2) & 1);
		offset += 8;
		
		for ( ; offset < buffer.byteLength; ) {
			switch (view.getUint8(offset++)) {
				case KEYWORDS.v:
					readTriplet('v');
					break;
				case KEYWORDS.vn:
					readTriplet('vn');
					break;
				case KEYWORDS.vt:
					readTriplet('vt');
					break;
				case KEYWORDS.g:
					readString('g');
					break;
				case KEYWORDS.o:
					readString('o');
					break;
				case KEYWORDS.s:
					readString('s');
					break;
				case KEYWORDS.f:
					const length = view.getUint32(offset);
					offset += 4;
					
					for (let i = 0; i < length; i++) {
						source += `f`;
						
						const L    = view.getUint8(offset++);
						const bits = view.getUint8(offset++);
						
						for (let j = 0; j < L; j++) {
							let triplet = [];
							for (let k = 0; k < 3; k++) {
								triplet.push(view[`getUint${bits}`](offset) || null);
								offset += bits / 8;
							}
							source += ` ${triplet.join('/').replace(/\/+$/, '')}`;
						}
						
						source += '\n';
					}
					break;
				default: throw new SyntaxError(`Invalid sequence at offset 0x${(offset-1).toString(16)}`);
			}
		}
		
		return source;
		
		
		
		function readString(keyword) {
			const length = view.getUint32(offset);
			offset += 4;
			
			const codes = Array(length);
			for (let i = 0; i < length; i++) {
				codes[i] = view.getUint16(offset);
				offset += 2;
			}
			
			source += `${keyword} ${String.fromCharCode.apply(null, codes)}\n`;
		}
		
		function readTriplet(keyword) {
			const length = view.getUint32(offset);
			offset += 4;
			
			for (let i = 0; i < length * 3; i++) {
				if (!(i % 3)) source += keyword;
				
				if (useFloat16) {
					source += ` ${getFloat16(view, offset)}`;
					offset += 2;
				} else {
					source += ` ${view.getFloat32(offset)}`;
					offset += 4;
				}
				
				if (i && (i % 3) === 2) source += '\n';
			}
		}
	} catch (e) {
		throw new SyntaxError(`File was not parsed correctly. Probably invalid data structure. Reason: ${e.message}`);
	}
}
