'use strict';

const {logError, logInfo} = require('./log');

const shutdownHandlers = {};

['SIGHUP', 'SIGINT', 'SIGTERM'].forEach(signal => {
	process.once(signal, async () => {
		logInfo(`Received ${signal}. Server is shutting down gracefully...`);
		await module.exports.handleShutdown();
		logInfo('Shutting down now...');
		process.kill(process.pid, signal);
	});
});

module.exports = {
	
	async handleShutdown() {
		const priorities = Object.keys(shutdownHandlers).sort((p1, p2) => p1 - p2);
		
		for (const priority of priorities) {
			for (const h of shutdownHandlers[priority]) {
				try {
					if (h.description) logInfo(` - ${h.description}`);
					await h.handler();
				} catch (e) {
					logError(e);
				}
			}
		}
	},
	
	registerShutdownHandler(handler, options) {
		options = Object.assign({}, {priority: 0}, options);
		
		if (!shutdownHandlers[options.priority]) shutdownHandlers[options.priority] = [];
		shutdownHandlers[options.priority].push({
			description: options.description,
			handler
		});
	},
	
	async shutdown(code) {
		logInfo('Server is shutting down gracefully...');
		await module.exports.handleShutdown();
		process.exit(code || 0);
	}
	
};
