'use strict';

const IS_EMPTY = Symbol('isEmpty');

module.exports = {
	
	SYMBOLS: {
		IS_EMPTY
	},
	
	handle(requestValidator, handler, responseValidator) {
		return async function (req, res, next) {
			const nextWatcher = new NextWatcher(next);
			const _next = nextWatcher.watcher.bind(nextWatcher);
			
			let response;
			
			if (requestValidator) {
				try {
					await requestValidator(req, res, _next);
					if (nextWatcher.wasCalled) return nextWatcher.doNext();
				} catch (e) { // Probably bad request
					return res.status(400).send(e.message);
				}
			}
			
			if (handler) {
				try {
					response = await handler(req, res, _next);
					if (nextWatcher.wasCalled) return nextWatcher.doNext();
				} catch (e) { // Operation error
					const code = e.code || (e.response && e.response.code);
					if (/^4\d\d/.test(code)) return res.status(code).send(e.message);
					
					return next(e);
				}
			}
			
			if (responseValidator) {
				try {
					response = await responseValidator(response, req, res, _next);
					if (nextWatcher.wasCalled) return nextWatcher.doNext();
				} catch (e) { // Probably bad response
					return next(e);
				}
			}
			
			if (!response && !res[IS_EMPTY]) return res.status(404).send(response);
			
			res.status(res.statusCode || 200).send(response);
		}
	},
	
	methodNotAllowed(router) {
		return function (req, res) {
			const layer = router.stack.filter(layer => layer.path === req.path)[0];
			if (!layer) return res.status(404).send();
			
			const routeMethods = Object.keys(layer.route.methods);
			const allowedMethods = routeMethods
				.filter(method => method[0] !== '_')
				.map(method => method.toUpperCase())
				.join(', ');
			
			return res.status(405).set('Allow', allowedMethods).send();
		};
	}
	
};

class NextWatcher {
	constructor(next) {
		this._next     = next;
		this.wasCalled = false;
		
	}
	
	doNext() {
		return this._next(...this._passedArgs);
	}
	
	watcher(...args) {
		this._passedArgs = args;
		this.wasCalled   = true;
	}
}
