'use strict';

const express = require('express');
const mongo   = require('mongodb');

const MongoClient = mongo.MongoClient;
const ObjectID    = mongo.ObjectID;

['DB_HOST', 'DB_NAME'].forEach(varname => {
	if (!process.env[varname]) {
		console.error(`Environment variable ${varname} is not defined!`);
		process.exit(1);
	}
});

const DB_HOST = process.env.DB_HOST;
const DB_NAME = process.env.DB_NAME;

const app = express();

app.use(express.json({
	limit: '16mb'
}));
app.use(function (req, res, next) {
	res.header('Access-Control-Allow-Origin', '*');
	res.header('Access-Control-Allow-Methods', 'POST, PUT, GET, PATCH, DELETE, OPTIONS');
	next();
});

app.post('/:collection', (req, res) => {
	if (isObjectEmpty(req.body)) {
		res.status(400).send('Body is empty');
		return;
	}
	
	mongoSafeConnect(res, client => {
		client.db(DB_NAME).collection(req.params.collection).insertOne(req.body, (err, result) => {
			if (err) throw err;
			
			const record = result.ops[0];
			res.status(201).append('Location', `/${req.params.collection}/${record._id}`).send(record);
			
			client.close();
		});
	});
});
app.put('/:collection/:id', (req, res) => {
	if (isObjectEmpty(req.body)) {
		res.status(400).send('Body is empty');
		return;
	}
	
	mongoSafeConnect(res, client => {
		const collection = client.db(DB_NAME).collection(req.params.collection);
		collection.findOne({_id: wrapId(req.params.id)}, (err, result) => {
			if (err) throw err;
			
			if (result) {
				res.status(409).send(`Record with key ${req.params.id} already exists`);
				client.close();
				return;
			}
			
			collection.insertOne({...req.body, _id: req.params.id}, (err, result) => {
				if (err) throw err;
				
				const record = result.ops[0];
				res.status(201).append('Location', `/${req.params.collection}/${record._id}`).send(record);
				
				client.close();
			});
		});
	});
});
app.get('/:collection', (req, res) => {
	mongoSafeConnect(res, client => {
		client.db(DB_NAME).collection(req.params.collection).find().toArray((err, result) => {
			if (err) throw err;
			
			res.status(200).send(result);
			
			client.close();
		});
	});
});
app.get('/:collection/:id', (req, res) => {
	mongoSafeConnect(res, client => {
		client.db(DB_NAME).collection(req.params.collection).findOne({_id: wrapId(req.params.id)}, (err, result) => {
			if (err) throw err;
			
			if (result) {
				res.status(200).send(result);
			} else {
				res.status(404).send();
			}
			
			client.close();
		});
	});
});
app.patch('/:collection/:id', (req, res) => {
	if (isObjectEmpty(req.body)) {
		res.status(400).send('Body is empty');
		return;
	}
	
	mongoSafeConnect(res, client => {
		delete req.body._id; // Immutable properties shouldn't be tried to change
		client.db(DB_NAME).collection(req.params.collection).findOneAndUpdate({_id: wrapId(req.params.id)}, {$set: req.body}, {returnOriginal: false}, (err, result) => {
			if (err) throw err;
			
			if (result.value) {
				res.status(200).send(result.value);
			} else {
				res.status(404).send();
			}
			
			client.close();
		});
	});
});
app.delete('/:collection/:id', (req, res) => {
	mongoSafeConnect(res, client => {
		client.db(DB_NAME).collection(req.params.collection).findOneAndDelete({_id: wrapId(req.params.id)}, (err, result) => {
			if (err) throw err;
			
			if (result.value) {
				res.status(200).send(result.value);
			} else {
				res.status(404).send();
			}
			
			client.close();
		});
	});
});

app.listen(80, () => {
	console.log('Server is started.');
});

function isObjectEmpty(obj) {
	for (const key in obj) {
		if (obj.hasOwnProperty(key)) return false;
	}
	
	return true;
}

function mongoSafeConnect(serverResponse, callback) {
	MongoClient.connect(DB_HOST, (err, client) => {
		try {
			if (err) throw err;
			
			if (callback && client) callback(client);
		} catch (e) {
			console.log(e);
			serverResponse.status(500).send();
			if (client) client.close();
		}
	});
}

function wrapId(id) {
	return ObjectID.isValid(id) ? new ObjectID(id) : id;
}
